<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;


class DashboardController extends Controller
{
    
    public function products()
    {
        $products = Product::latest()->get();
        return view('admin.products.index', compact('products'));
    }

    
    public function create()
    {
        return view('admin.products.create');
    }

    
    public function store(Request $request)
    {
        $request->validate([
            'name'        => 'required|string',
            'description' => 'nullable|string',
            'image'       => 'nullable|string',
            'price'       => 'required|numeric',
            'stock'       => 'required|integer',
        ]);

        Product::create([
            'name'        => $request->name,
            'description' => $request->description,
            'image'       => $request->image,
            'price'       => $request->price,
            'stock'       => $request->stock,
        ]);

        return redirect()->route('admin.products')
            ->with('success', 'Produk berhasil ditambahkan');
    }

    
    public function edit($id)
    {
        $product = Product::findOrFail($id);
        return view('admin.products.edit', compact('product'));
    }

    
    public function update(Request $request, $id)
    {
        $request->validate([
            'name'        => 'required|string',
            'description' => 'nullable|string',
            'image'       => 'nullable|string',
            'price'       => 'required|numeric',
            'stock'       => 'required|integer',
        ]);

        $product = Product::findOrFail($id);

        $product->update([
            'name'        => $request->name,
            'description' => $request->description,
            'image'       => $request->image,
            'price'       => $request->price,
            'stock'       => $request->stock,
        ]);

        return redirect()->route('admin.products')
            ->with('success', 'Produk berhasil diperbarui');
    }

    
    public function destroy($id)
    {
        Product::findOrFail($id)->delete();

        return redirect()->route('admin.products')
            ->with('success', 'Produk berhasil dihapus');
    }

     

public function adminsIndex()
{
    $admins = DB::table('admins')->latest()->get();
    return view('admin.admins.index', compact('admins'));
}

public function adminsCreate()
{
    return view('admin.admins.create');
}

public function adminsStore(Request $request)
{
    $request->validate([
        'name'     => 'required|string',
        'email'    => 'required|email|unique:admins,email',
        'password' => 'required|min:6',
    ]);

    DB::table('admins')->insert([
        'name'       => $request->name,
        'email'      => $request->email,
        'password'   => Hash::make($request->password),
        'created_at' => now(),
        'updated_at' => now(),
    ]);

    return redirect()->route('admin.admins.index')
        ->with('success', 'Admin berhasil ditambahkan');
}

public function adminsEdit($id)
{
    $admin = DB::table('admins')->where('id', $id)->first();
    return view('admin.admins.edit', compact('admin'));
}

public function adminsUpdate(Request $request, $id)
{
    $request->validate([
        'name'  => 'required|string',
        'email' => 'required|email|unique:admins,email,' . $id,
    ]);

    $data = [
        'name'       => $request->name,
        'email'      => $request->email,
        'updated_at' => now(),
    ];

    if ($request->filled('password')) {
        $data['password'] = Hash::make($request->password);
    }

    DB::table('admins')->where('id', $id)->update($data);

    return redirect()->route('admin.admins.index')
        ->with('success', 'Admin berhasil diperbarui');
}

public function adminsDestroy($id)
{
    DB::table('admins')->where('id', $id)->delete();

    return redirect()->route('admin.admins.index')
        ->with('success', 'Admin berhasil dihapus');
}
}
